<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ArticleResource\Pages;
use App\Models\Article;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Str;

// Import Komponen
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\TagsInput; // <--- Import Ini
use Filament\Forms\Components\Section;

class ArticleResource extends Resource
{
    protected static ?string $model = Article::class;
    protected static ?string $navigationIcon = 'heroicon-o-newspaper';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Konten Utama')
                    ->schema([
                        TextInput::make('title')
                            ->live(onBlur: true)
                            ->afterStateUpdated(fn($set, ?string $state) => $set('slug', Str::slug($state)))
                            ->required(),
                        TextInput::make('slug')->required()->readOnly(),

                        TextInput::make('category')->label('Kategori')->required(),

                        // INPUT TAGS BARU
                        TagsInput::make('tags')
                            ->label('Tags (Tekan Enter)')
                            ->separator(',')
                            ->placeholder('Contoh: Teknologi, Coding, Laravel'),
                        FileUpload::make('thumbnail')
                            ->label('Gambar Utama')
                            ->image() // Wajib file gambar
                            ->disk('public') // Simpan di folder public
                            ->directory('articles')
                            ->visibility('public') // Bisa dilihat umum
                            ->preserveFilenames(false) // Acak nama file (Mencegah error nama kembar)
                            ->imageEditor() // BONUS: Tambah fitur crop/edit gambar
                            ->required()
                            ->columnSpanFull(),
                        RichEditor::make('content')->required()->columnSpanFull(),
                        Toggle::make('is_published')->label('Publish Now'),
                    ])
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\ImageColumn::make('thumbnail'),
                Tables\Columns\TextColumn::make('title')->searchable(),
                Tables\Columns\TextColumn::make('category')->badge(),
                Tables\Columns\TextColumn::make('created_at')->date(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListArticles::route('/'),
            'create' => Pages\CreateArticle::route('/create'),
            'edit' => Pages\EditArticle::route('/{record}/edit'),
        ];
    }
}