<?php

namespace App\Filament\Resources;

use App\Filament\Resources\UserResource\Pages;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Hash; // PENTING: Untuk enkripsi password

// Komponen Form
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Section;

// Komponen Table
use Filament\Tables\Columns\TextColumn;

class UserResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-users'; // Icon User
    
    // Masukkan ke grup yang sama agar jadi satu dropdown
    protected static ?string $navigationGroup = 'Content Management'; 
    
    protected static ?string $navigationLabel = 'Manajemen User'; // Label di Menu
    protected static ?int $navigationSort = 3; // Urutan ke-3 di dropdown

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Data Pengguna')
                    ->schema([
                        TextInput::make('name')
                            ->label('Nama Lengkap')
                            ->required()
                            ->maxLength(255),

                        TextInput::make('email')
                            ->email()
                            ->required()
                            ->maxLength(255)
                            ->unique(ignoreRecord: true), // Cek unik kecuali punya sendiri saat edit

                        TextInput::make('password')
                            ->password()
                            ->label('Password')
                            // Hanya wajib diisi saat membuat user baru (Create)
                            ->required(fn (string $operation): bool => $operation === 'create')
                            // Hash password sebelum simpan ke DB
                            ->dehydrateStateUsing(fn ($state) => Hash::make($state))
                            // Jangan update password jika form kosong saat edit
                            ->dehydrated(fn ($state) => filled($state))
                            ->revealable(), // Tombol mata untuk lihat password
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->label('Nama')
                    ->searchable(),
                
                TextColumn::make('email')
                    ->searchable(),
                
                TextColumn::make('created_at')
                    ->label('Bergabung')
                    ->dateTime('d M Y')
                    ->sortable(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(), // Bisa edit user
                Tables\Actions\DeleteAction::make(), // Bisa hapus user
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }
}