<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PortfolioResource\Pages;
use App\Filament\Resources\PortfolioResource\RelationManagers;
use App\Models\Portfolio;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

// --- IMPORT KOMPONEN FORM ---
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Section; // Untuk layout yang lebih rapi

// --- IMPORT KOMPONEN TABLE ---
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Filters\SelectFilter;

class PortfolioResource extends Resource
{
    protected static ?string $model = Portfolio::class;

    protected static ?string $navigationIcon = 'heroicon-o-camera'; // Icon Kamera
    protected static ?string $navigationLabel = 'Portofolio';
    protected static ?string $navigationGroup = 'Content Management'; // Grouping menu
    protected static ?int $navigationSort = 2;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                // Kita bungkus dalam Section agar tampilan putih rapi
                Section::make('Informasi Proyek')
                    ->description('Masukkan detail proyek portofolio perusahaan.')
                    ->schema([
                        TextInput::make('title')
                            ->label('Judul Proyek')
                            ->required()
                            ->maxLength(255),

                        Select::make('category')
                            ->label('Kategori')
                            ->options([
                                'Website' => 'Website Development',
                                'Mobile' => 'Mobile App',
                                'Branding' => 'Branding & Design',
                                'SEO' => 'SEO & Marketing',
                            ])
                            ->required()
                            ->native(false) // Tampilan dropdown modern
                            ->searchable(),

                        TextInput::make('client')
                            ->label('Nama Klien')
                            ->placeholder('Contoh: PT. Maju Mundur')
                            ->maxLength(255),

                        DatePicker::make('project_date')
                            ->label('Tanggal Proyek')
                            ->native(false), // Menggunakan datepicker JS yang bagus
                    ])->columns(2), // Grid 2 kolom

                Section::make('Media & Detail')
                    ->schema([
                        FileUpload::make('image')
                            ->label('Gambar Utama')
                            ->image() // Wajib file gambar
                            ->disk('public') // Simpan di folder public
                            ->directory('portfolios') // Dalam folder portfolios
                            ->visibility('public') // Bisa dilihat umum
                            ->preserveFilenames(false) // Acak nama file (Mencegah error nama kembar)
                            ->imageEditor() // BONUS: Tambah fitur crop/edit gambar
                            ->required()
                            ->columnSpanFull(),

                        RichEditor::make('description')
                            ->label('Deskripsi Lengkap')
                            ->required()
                            ->columnSpanFull()
                            ->toolbarButtons([
                                'bold',
                                'italic',
                                'bulletList',
                                'orderedList',
                                'link',
                                'h2',
                                'h3',
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                ImageColumn::make('image')
                    ->label('Cover')
                    ->square(),

                TextColumn::make('title')
                    ->label('Judul')
                    ->searchable()
                    ->sortable()
                    ->weight('bold'),

                TextColumn::make('category')
                    ->badge() // Tampil sebagai badge warna-warni
                    ->color(fn(string $state): string => match ($state) {
                        'Website' => 'info',   // Biru
                        'Mobile' => 'success', // Hijau
                        'Branding' => 'warning', // Kuning
                        'SEO' => 'danger',     // Merah
                        default => 'gray',
                    })
                    ->sortable(),

                TextColumn::make('client')
                    ->label('Klien')
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: false), // Bisa disembunyikan user

                TextColumn::make('project_date')
                    ->label('Tanggal')
                    ->date('d M Y')
                    ->sortable(),
            ])
            ->defaultSort('created_at', 'desc') // Urutkan dari yang terbaru
            ->filters([
                // Tambahkan Filter agar Admin bisa memilah data berdasarkan Kategori
                SelectFilter::make('category')
                    ->label('Filter Kategori')
                    ->options([
                        'Website' => 'Website Development',
                        'Mobile' => 'Mobile App',
                        'Branding' => 'Branding & Design',
                        'SEO' => 'SEO & Marketing',
                    ]),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPortfolios::route('/'),
            'create' => Pages\CreatePortfolio::route('/create'),
            'edit' => Pages\EditPortfolio::route('/{record}/edit'),
        ];
    }
}